package dev.gegy.mdchat;

import com.google.common.collect.Lists;
import dev.gegy.mdchat.parser.ColoredChatExtension;
import dev.gegy.mdchat.parser.FormattedNode;
import dev.gegy.mdchat.parser.SpoilerExtension;
import dev.gegy.mdchat.parser.SpoilerNode;
import net.fabricmc.fabric.api.event.Event;
import net.fabricmc.fabric.api.event.EventFactory;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_124;
import net.minecraft.class_2558;
import net.minecraft.class_2568;
import net.minecraft.class_2583;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_5250;
import net.minecraft.text.*;
import org.commonmark.ext.autolink.AutolinkExtension;
import org.commonmark.ext.gfm.strikethrough.Strikethrough;
import org.commonmark.ext.gfm.strikethrough.StrikethroughExtension;
import org.commonmark.node.*;
import org.commonmark.parser.Parser;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;

public final class TextStyler {
    public static final TextStyler INSTANCE = ofGlobal();

    private static final class_2583 SPOILER = class_2583.field_24360.method_27705(class_124.field_1063, class_124.field_1051);

    private final Parser PARSER;
    private final Event<NodeStyler> STYLER;

    private TextStyler(Parser parser, Event<NodeStyler> styler) {
        this.PARSER = parser;
        this.STYLER = styler;
    }

    /**
     * Creates a new TextStyler based off of globally-invoked bootstraps.
     *
     * @return A global-like TextStyler.
     */
    // TODO: Consider either passing in builders or a bootstrap?
    public static TextStyler ofGlobal() {
        Parser.Builder parserBuilder = Parser.builder()
                .enabledBlockTypes(Collections.emptySet())
                .extensions(Lists.newArrayList(
                        ColoredChatExtension.INSTANCE,
                        SpoilerExtension.INSTANCE,
                        AutolinkExtension.create(),
                        StrikethroughExtension.create()
                ));
        Event<NodeStyler> eventBus = EventFactory.createArrayBacked(NodeStyler.class, NodeStyler.EVENT_INVOKER);
        // We're using FabricLoader's entrypoint system here as it isn't possible to
        // determine when we'll be bootstrapped as any mod can load this class at any time.
        for (var bootstrap : FabricLoader.getInstance().getEntrypoints("markdown-chat", StylerBootstrap.class)) {
            bootstrap.bootstrap(parserBuilder, eventBus);
        }
        return new TextStyler(parserBuilder.build(), eventBus);
    }

    /**
     * Creates a new TextStyler using the provided parser and event bus.
     *
     * @param parser The completed parser to use.
     * @param styler The completed styler to use.
     * @return A TextStyler tuned to the needs of the input.
     */
    public static TextStyler ofLocal(Parser parser, Event<NodeStyler> styler) {
        return new TextStyler(parser, styler);
    }

    @Nullable
    public net.minecraft.class_2561 apply(String string) {
        Node node = PARSER.parse(string);
        return this.renderAsText(node);
    }

    @Nullable
    private class_5250 renderAsText(Node node) {
        // FIXME: Perhaps should be a lazy supplier?
        class_5250 text = STYLER.invoker().style(node, () -> renderChildren(node));
        if (text != null) {
            return text;
        }

        // TODO: Consider perhaps making this its own invocation,
        //  or just leave as is for fallbacks?
        if (node instanceof Text) {
            return this.renderLiteral((Text) node);
        } else if (node instanceof Code) {
            return this.renderCode((Code) node);
        } else if (node instanceof StrongEmphasis) {
            return this.renderStrongEmphasis((StrongEmphasis) node);
        } else if (node instanceof Emphasis) {
            return this.renderEmphasis(node, class_124.field_1056);
        } else if (node instanceof Strikethrough) {
            return this.renderEmphasis(node, class_124.field_1055);
        } else if (node instanceof Link) {
            return this.renderLink((Link) node);
        } else if (node instanceof FormattedNode) {
            return this.renderFormattedText((FormattedNode) node);
        } else if (node instanceof SpoilerNode) {
            return this.renderSpoiler((SpoilerNode) node);
        }

        return this.renderChildren(node);
    }

    private class_5250 renderLiteral(Text text) {
        return new class_2585(text.getLiteral());
    }

    private class_5250 renderCode(Code code) {
        String literal = code.getLiteral();
        class_5250 text = new class_2585(literal).method_27692(class_124.field_1080);
        if (literal.startsWith("/")) {
            return text.method_27694(style -> style
                    .method_10949(new class_2568(class_2568.class_5247.field_24342, new class_2585("Click to Copy to Console")))
                    .method_10958(new class_2558(class_2558.class_2559.field_11745, literal))
            );
        } else {
            return text.method_27694(style -> style
                    .method_10949(new class_2568(class_2568.class_5247.field_24342, new class_2588("chat.copy.click")))
                    .method_10958(new class_2558(class_2558.class_2559.field_21462, literal))
            );
        }
    }

    private class_5250 renderStrongEmphasis(StrongEmphasis emphasis) {
        String delimiter = emphasis.getOpeningDelimiter();
        if (delimiter.equals("__")) {
            return this.renderEmphasis(emphasis, class_124.field_1073);
        } else {
            return this.renderEmphasis(emphasis, class_124.field_1067);
        }
    }

    @Nullable
    private class_5250 renderFormattedText(FormattedNode formatted) {
        class_5250 text = this.renderChildren(formatted);
        if (text != null) {
            return text.method_27692(formatted.getFormatting());
        }
        return null;
    }

    @Nullable
    private class_5250 renderSpoiler(SpoilerNode spoiler) {
        class_5250 text = this.renderChildren(spoiler);
        if (text != null) {
            return text.method_27661()
                    .method_10862(SPOILER.method_10949(new class_2568(class_2568.class_5247.field_24342, text)));
        }
        return null;
    }

    @Nullable
    private class_5250 renderEmphasis(Node node, class_124 formatting) {
        class_5250 text = this.renderChildren(node);
        if (text != null) {
            return text.method_27692(formatting);
        }
        return null;
    }

    @Nullable
    private class_5250 renderLink(Link link) {
        class_5250 title = this.renderChildren(link);

        if (title == null) {
            title = new class_2585(link.getDestination());
        }

        class_5250 redirectsTo = new class_2585("Goes to ")
                .method_10852(new class_2585(link.getDestination()).method_27695(class_124.field_1075, class_124.field_1073))
                .method_27695(class_124.field_1080, class_124.field_1056);

        String hoverText = link.getTitle();
        class_5250 hover;
        if (hoverText != null) {
            hover = new class_2585(hoverText).method_27693("\n\n").method_10852(redirectsTo);
        } else {
            hover = redirectsTo;
        }

        return title.method_10862(this.buildLinkStyle(link.getDestination(), hover));
    }

    private class_2583 buildLinkStyle(String url, class_5250 hover) {
        return class_2583.field_24360
                .method_27705(class_124.field_1075, class_124.field_1073)
                .method_10958(new class_2558(class_2558.class_2559.field_11749, url))
                .method_10949(new class_2568(class_2568.class_5247.field_24342, hover));
    }

    @Nullable
    private class_5250 renderChildren(Node parent) {
        class_5250 result = null;

        Node child = parent.getFirstChild();
        while (child != null) {
            Node next = child.getNext();

            class_5250 text = this.renderAsText(child);
            if (text != null) {
                if (result == null) {
                    result = new class_2585("");
                }
                result = result.method_10852(text);
            }

            child = next;
        }

        return result;
    }
}
